/**
  ******************************************************************************
  * @file    main.c
  * @author  B.VERNOUX
  * @Version V0.1.0
  * @Date    2023-08-26
  * @brief   main function
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include <stdlib.h>
#include "main.h"	
#include "string.h"

#include "apps_common.h"
#include "llcc68.h"
#include "llcc68_hal.h"
//#include "main_ping_pong.h"
#include "smtc_hal_mcu.h"
#include "smtc_hal_dbg_trace.h"

//extern UART_HandleTypeDef huart1;

/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private user code ---------------------------------------------------------*/

/*
# LLCC68 Ping Pong example

## Description

The application will automatically set the device in Ping-Pong mode.

The sample code will be used to perform test for both LoRa and FSK modem tests. Define macro `PACKET_TYPE` to `SX126X_PKT_TYPE_LORA` or `SX126X_PKT_TYPE_GFSK` (in file (`../../common/apps_configuration.h`)) to enable each modem in the test.

The application can make device work as master or slave in Ping-Pong test. When starting up, it assumes to be a master and will send 'PING' packets and receive 'PONG' packets alternatively. When a 'PING' packet received, the device switches itself to slave mode, then send 'PONG' packets and expect to receive 'PING' packets. In slave mode, if the device receive a 'non-PING' packet, it will reset to master mode and start the previous process again.

## Configuration

Several parameters can be updated in `../../common/apps_configuration.h` header file, refer to `../../common/README.md` for more details.

Several parameters can be updated in `main_ping_pong.h` header file:

| Constant           | Comments                    | Default value                | Default value |
| ------------------ | --------------------------- | ---------------------------- | ------------- |
| `RX_TIMEOUT_VALUE` | timeout value for reception | Any value that fits uint32_t | 600           |

*/

#define RX_TIMEOUT_VALUE 600

/**
 * @brief Size of ping-pong message prefix
 *
 * Expressed in bytes
 */
#define PING_PONG_PREFIX_SIZE 4

/**
 * @brief Threshold:number of exchanges before informing user on UART that the board pair is still not synchronized
 *
 * Expressed in number of packet exchanged
 */
#define SYNC_PACKET_THRESHOLD 64

/**
 * @brief Number of exchanges are stored in the payload of the packet exchanged during this PING-PONG communication
 *        this constant indicates where in the packet the two bytes used to count are located
 *
 * Expressed in bytes
 */
#define ITERATION_INDEX ( PING_PONG_PREFIX_SIZE + 1 )

/**
 * @brief Duration of the wait between each ping-pong activity, can be used to adjust ping-pong speed
 *
 * Expressed in milliseconds
 */
#define DELAY_PING_PONG_PACE_MS 200

/**
 * @brief Duration of the wait before packet transmission to assure reception status is ready on the other side
 *
 * Expressed in milliseconds
 */
#define DELAY_BEFORE_TX_MS 20

static uint8_t buffer_tx[PAYLOAD_LENGTH];
static bool    is_master = true;

const uint8_t ping_msg[PING_PONG_PREFIX_SIZE] = "PING";
const uint8_t pong_msg[PING_PONG_PREFIX_SIZE] = "PONG";

static uint16_t iteration       = 0;
static uint16_t packets_to_sync = 0;
static void* context;

void loop_tests(void)
{
	printf("loop 6M nop start\n");
	for(int i=0; i < 6000000; i++)
	{
		asm("nop");
	}
	printf("loop 6M nop end\n");
}

/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	static volatile int start = 0;
	uint8_t sync_word[2] = { 0 };
	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	HAL_Init();
	
	/* Configure the system clock to HIRC 24MHz*/
	SystemClock_Config();

	/* Init BSP */
	BSP_GPIO_Init();
	BSP_SPI_Init();

	/* Configure uart1 for printf */
	LogInit();
	printf("LLCC68 Ping Pong logs UART1, PA2-TXD, PA1-RXD\n");

	/* Debug point */
	/*
	do{

	}while(start != 1);
	*/

    HAL_DBG_TRACE_INFO("===== LLCC68 Ping Pong =====\n\n");
    apps_common_llcc68_print_version_info();

    apps_common_shield_init();
    context = apps_common_llcc68_get_context();

    apps_common_llcc68_init(context);

    llcc68_get_lora_sync_word(context, (uint8_t*)&sync_word);
    HAL_DBG_TRACE_INFO("LLCC68 get default lora syncword=0x%02X%02X (expected 0x1424)\n\n", sync_word[0], sync_word[1]);

    apps_common_llcc68_radio_init(context);

    llcc68_set_dio_irq_params(
        context, LLCC68_IRQ_ALL,
        LLCC68_IRQ_TX_DONE | LLCC68_IRQ_RX_DONE | LLCC68_IRQ_TIMEOUT | LLCC68_IRQ_HEADER_ERROR | LLCC68_IRQ_CRC_ERROR,
        LLCC68_IRQ_NONE, LLCC68_IRQ_NONE);

    llcc68_clear_irq_status(context, LLCC68_IRQ_ALL);

    /* Initializes random number generator */
    srand(10);
    memcpy(buffer_tx, ping_msg, PING_PONG_PREFIX_SIZE);
    buffer_tx[PING_PONG_PREFIX_SIZE] = 0;
    buffer_tx[ITERATION_INDEX]       = (uint8_t)(iteration);
    for(int i = PING_PONG_PREFIX_SIZE + 1 + 1; i < PAYLOAD_LENGTH; i++)
    {
        buffer_tx[i] = i;
    }

    llcc68_write_buffer(context, 0, buffer_tx, PAYLOAD_LENGTH);

    apps_common_llcc68_handle_pre_tx();
    llcc68_set_tx(context, 0);

    while( 1 )
    {
        apps_common_llcc68_irq_process(context);
    }
/*
	while (1)
	{
		HAL_Delay(1000);
		printf("UART1(PA2-TXD, PA1-RXD) SysClockFreq %d MHz\n", HAL_RCC_GetSysClockFreq()/1000000);
		//loop_tests();
	}
*/
}

void on_tx_done( void )
{
    apps_common_llcc68_handle_post_tx( );
    HAL_DBG_TRACE_INFO( "Sent message %s, iteration %d\n", buffer_tx, iteration );

    HAL_Delay(DELAY_PING_PONG_PACE_MS);

    apps_common_llcc68_handle_pre_rx( );
    llcc68_set_rx( context, get_time_on_air_in_ms( ) + RX_TIMEOUT_VALUE +
                                rand( ) % 500 );  // Random delay to avoid unexpected sync
}

void on_rx_done( void )
{
    uint8_t buffer_rx[PAYLOAD_LENGTH];
    uint8_t size;

    packets_to_sync = 0;

    apps_common_llcc68_handle_post_rx( );

    apps_common_llcc68_receive(context, buffer_rx, &size, PAYLOAD_LENGTH );
    iteration = buffer_rx[ITERATION_INDEX];

    iteration++;
    HAL_DBG_TRACE_INFO( "Received message %s, iteration %d\n", buffer_rx, iteration );

    if( is_master == true )
    {
        if( memcmp( buffer_rx, ping_msg, PING_PONG_PREFIX_SIZE ) == 0 )
        {
            is_master = false;
            memcpy( buffer_tx, pong_msg, PING_PONG_PREFIX_SIZE );
        }
        else if( memcmp( buffer_rx, pong_msg, PING_PONG_PREFIX_SIZE ) != 0 )
        {
            HAL_DBG_TRACE_ERROR( "Unexpected message - PONG expected\n" );
        }
    }
    else
    {
        if( memcmp( buffer_rx, ping_msg, PING_PONG_PREFIX_SIZE ) != 0 )
        {
            HAL_DBG_TRACE_ERROR( "Unexpected message\n" );

            is_master = true;
            memcpy( buffer_tx, ping_msg, PING_PONG_PREFIX_SIZE );
        }
    }

    HAL_Delay( DELAY_PING_PONG_PACE_MS + DELAY_BEFORE_TX_MS );
    buffer_tx[ITERATION_INDEX] = ( uint8_t ) ( iteration );

    llcc68_write_buffer( context, 0, buffer_tx, PAYLOAD_LENGTH );

    apps_common_llcc68_handle_pre_tx( );
    llcc68_set_tx( context, 0 );
}

void on_rx_timeout( void )
{
    packets_to_sync++;
    if( packets_to_sync > SYNC_PACKET_THRESHOLD )
    {
        HAL_DBG_TRACE_WARNING(
            "It looks like synchronisation is still not done, consider resetting one of the board\n" );
    }
    apps_common_llcc68_handle_post_rx( );

    is_master = true;
    iteration = 0;
    memcpy( buffer_tx, ping_msg, PING_PONG_PREFIX_SIZE );
    buffer_tx[ITERATION_INDEX] = ( uint8_t ) ( iteration );

    llcc68_write_buffer( context, 0, buffer_tx, PAYLOAD_LENGTH );

    apps_common_llcc68_handle_pre_tx( );
    llcc68_set_tx( context, 0 );
}

void on_rx_error( void )
{
    apps_common_llcc68_handle_post_rx( );

    is_master = true;
    iteration = 0;
    memcpy( buffer_tx, ping_msg, PING_PONG_PREFIX_SIZE );
    buffer_tx[ITERATION_INDEX] = ( uint8_t ) ( iteration );

    llcc68_write_buffer( context, 0, buffer_tx, PAYLOAD_LENGTH );

    apps_common_llcc68_handle_pre_tx( );
    llcc68_set_tx( context, 0 );
}


/**
  * @brief System Clock Configuration
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_OscInitTypeDef RCC_OscInitStruct = {0};
	RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

	RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HIRC;
	RCC_OscInitStruct.HIRCState = RCC_HIRC_ON;
	RCC_OscInitStruct.HIRCCalibrationValue = RCC_HIRCCALIBRATION_24M;
	
	if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
	{
		Error_Handler();
	}
	
	/**Initializes the CPU, AHB and APB busses clocks
	*/
	RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK;
	RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_HIRC;
	RCC_ClkInitStruct.AHBCLKDivider = RCC_HCLK_DIV1;
	RCC_ClkInitStruct.APBCLKDivider = RCC_PCLK_DIV1;
	
	if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct) != HAL_OK)
	{
		Error_Handler();
	}
}
 

/**
  * @brief  This function is executed in case of error occurrence.
  * @retval None
  */
void Error_Handler(void)
{
  /* USER CODE BEGIN Error_Handler_Debug */
  /* User can add his own implementation to report the HAL error return state */

  /* USER CODE END Error_Handler_Debug */
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{ 
  /* USER CODE BEGIN 6 */
  /* User can add his own implementation to report the file name and line number,
     tex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* USER CODE END 6 */
}
#endif /* USE_FULL_ASSERT */


/* Private function -------------------------------------------------------*/



