/**
 * @file      llcc68_hal.c
 *
 * @brief     Implements the LLCC68 radio HAL functions
 */

/*
 * -----------------------------------------------------------------------------
 * --- DEPENDENCIES ------------------------------------------------------------
 */

#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>
#include "llcc68_hal.h"
#include "cx32l003_ebyte_e220_uart.h"
#include "apps_configuration.h"
/*
#include "smtc_hal_mcu_spi.h"
#include "smtc_hal_mcu_gpio.h"
*/
/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC MACROS -----------------------------------------------------------
 */

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC CONSTANTS --------------------------------------------------------
 */

const smtc_shield_llcc68_pa_pwr_cfg_t smtc_shield_llcc68_pa_pwr_cfg_table[SMTC_SHIELD_LLCC68_MAX_PWR - SMTC_SHIELD_LLCC68_MIN_PWR + 1] = {
    {   // Expected output power = -9dBm
        .power = 2,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -8dBm
        .power = 5,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -7dBm
        .power = 5,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -6dBm
        .power = 8,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -5dBm
        .power = 3,
        .pa_config = {
            .hp_max        = 0x02,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -4dBm
        .power = 9,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -3dBm
        .power = 10,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -2dBm
        .power = 11,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = -1dBm
        .power = 13,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 0dBm
        .power = 19,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 1dBm
        .power = 16,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 2dBm
        .power = 20,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 3dBm
        .power = 18,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x03,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 4dBm
        .power = 21,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 5dBm
        .power = 16,
        .pa_config = {
            .hp_max        = 0x02,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 6dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 7dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 8dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x02,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 9dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x03,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 10dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x01,
            .pa_duty_cycle = 0x04,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 11dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x02,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 12dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x02,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 13dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x02,
            .pa_duty_cycle = 0x02,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 14dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x02,
            .pa_duty_cycle = 0x03,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 15dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x03,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 16dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x03,
            .pa_duty_cycle = 0x02,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 17dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x05,
            .pa_duty_cycle = 0x00,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 18dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x05,
            .pa_duty_cycle = 0x01,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 19dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x05,
            .pa_duty_cycle = 0x02,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 20dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x06,
            .pa_duty_cycle = 0x03,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 21dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x06,
            .pa_duty_cycle = 0x04,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
    { // Expected output power = 22dBm
        .power = 22,
        .pa_config = {
            .hp_max        = 0x07,
            .pa_duty_cycle = 0x04,
            .device_sel    = 0x00,
            .pa_lut        = 0x01,
        },
    },
};

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC TYPES ------------------------------------------------------------
 */

/*
 * -----------------------------------------------------------------------------
 * --- PRIVATE FUNCTIONS DECLARATION -------------------------------------------
 */
/**
 * @brief Wait until radio busy pin returns to 0
 */
static void llcc68_hal_wait_on_busy(void)
{
	GPIO_PinState gpio_state;
    do
    {
    	gpio_state = HAL_GPIO_ReadPin(LLCC68_BUSY_GPIO_PORT, LLCC68_BUSY_GPIO_PIN);
    } while( gpio_state == GPIO_PIN_SET);
}

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC FUNCTIONS PROTOTYPES ---------------------------------------------
 */

bool smtc_shield_llcc68_is_dio2_set_as_rf_switch(void)
{
    return true;
}

llcc68_reg_mod_t smtc_shield_llcc68_get_reg_mode(void)
{
    return LLCC68_REG_MODE_LDO;
}

const smtc_shield_llcc68_pa_pwr_cfg_t* smtc_shield_llcc68_get_pa_pwr_cfg(const uint32_t rf_freq_in_hz,
                                                                         int8_t expected_output_pwr_in_dbm )
{
    if( ( SMTC_SHIELD_LLCC68_FREQ_MIN <= rf_freq_in_hz ) && ( rf_freq_in_hz <= SMTC_SHIELD_LLCC68_FREQ_MAX ) )
    {
        if( ( SMTC_SHIELD_LLCC68_MIN_PWR <= expected_output_pwr_in_dbm ) &&
            ( expected_output_pwr_in_dbm <= SMTC_SHIELD_LLCC68_MAX_PWR ) )
        {
            return &(
                smtc_shield_llcc68_pa_pwr_cfg_table[expected_output_pwr_in_dbm - SMTC_SHIELD_LLCC68_MIN_PWR] );
        }
    }
    return NULL;
}

llcc68_hal_status_t llcc68_hal_reset(const void* context)
{
    SPI_HandleTypeDef* hal_hspi = (SPI_HandleTypeDef*)context;

    HAL_GPIO_WritePin(LLCC68_SX_NRESET_GPIO_PORT, LLCC68_SX_NRESET_GPIO_PIN, GPIO_PIN_RESET); // GPIO_PIN_RESET = LOW
    HAL_Delay(1);
    HAL_GPIO_WritePin(LLCC68_SX_NRESET_GPIO_PORT, LLCC68_SX_NRESET_GPIO_PIN, GPIO_PIN_SET); // GPIO_PIN_SET = HIGH

    return LLCC68_HAL_STATUS_OK;
}

llcc68_hal_status_t llcc68_hal_wakeup(const void* context)
{
    SPI_HandleTypeDef* hal_hspi = (SPI_HandleTypeDef*)context;

	HAL_SPI_Set_NSS(hal_hspi, SPI_NSS_MODE_LOW);
    HAL_Delay(1);
	HAL_SPI_Set_NSS(hal_hspi, SPI_NSS_MODE_HIGH);

    return LLCC68_HAL_STATUS_OK;
}

llcc68_hal_status_t llcc68_hal_write(const void* context, const uint8_t* command, const uint16_t command_length,
                                     const uint8_t* data, const uint16_t data_length)
{
	uint16_t i;
    SPI_HandleTypeDef* hal_hspi = (SPI_HandleTypeDef*)context;
	uint8_t r_data;

    llcc68_hal_wait_on_busy();

	HAL_SPI_Set_NSS(hal_hspi, SPI_NSS_MODE_LOW);

	/* SPI Write command */
	for(i = 0; i < command_length; i++)
	{
		HAL_SPI_Master_Send_Data(hal_hspi, command[i]);
	}
	HAL_SPI_Master_Receive_Data(hal_hspi, &r_data); // Flush last data

	/* SPI Write data */
	for(i = 0; i < data_length; i++)
	{
		HAL_SPI_Master_Send_Data(hal_hspi, data[i]);
	}
	HAL_SPI_Master_Receive_Data(hal_hspi, &r_data); // Flush last data

	HAL_SPI_Set_NSS(hal_hspi, SPI_NSS_MODE_HIGH);

    return LLCC68_HAL_STATUS_OK;
}

llcc68_hal_status_t llcc68_hal_read(const void* context, const uint8_t* command, const uint16_t command_length,
                                    uint8_t* data, const uint16_t data_length)
{
	uint16_t i;
    SPI_HandleTypeDef* hal_hspi = (SPI_HandleTypeDef*)context;
	uint8_t r_data;
	uint8_t w_data = 0;

    llcc68_hal_wait_on_busy();

	HAL_SPI_Set_NSS(hal_hspi, SPI_NSS_MODE_LOW);

	/* SPI Write command */
	for(i = 0; i < command_length; i++)
	{
		HAL_SPI_Master_Send_Data(hal_hspi, command[i]);
	}
	HAL_SPI_Master_Receive_Data(hal_hspi, &r_data); // Flush last data

	/* SPI Read data */
	for(i = 0; i < data_length; i++)
	{
		HAL_SPI_Master_Send_Data(hal_hspi, w_data);
		HAL_SPI_Master_Receive_Data(hal_hspi, &data[i]);
	}

	HAL_SPI_Set_NSS(hal_hspi, SPI_NSS_MODE_HIGH);

    return LLCC68_HAL_STATUS_OK;
}

/* --- EOF ------------------------------------------------------------------ */
