/**
******************************************************************************
* @file    cx32l003_ebyte_e220_uart.c
* @author  B.VERNOUX
* @Version V1.0.0
* @Date    04-Sep-2023
* @brief  This file contains definitions for:
*          - UART1 / SPI1(LLCC68) / GPIO on EBYTE E220 UART board (like E220-900T22D with CX32L003F8)
*
@verbatim
**/

/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include "cx32l003_ebyte_e220_uart.h"

/** @addtogroup BSP
  * @{
  */ 

/** @defgroup CX32L003_EBYTE_E220_UART CX32L003 EBYTE_E220_UART
  * @brief This file provides set of firmware functions to manage UART / GPIO / SPI(LLCC68)
  *        available on EBYTE E220 UART board (like E220-900T22D with CX32L003F8).
  * @{
  */ 

/** @defgroup CX32L003_EBYTE_E220_UART_Private_Defines CX32L003 EBYTE_E220_UART Private Defines
  * @{
  */ 
  
/**
* @brief CX32L003 EBYTE_E220_UART BSP Driver version
*/
#define __CX32L003_EBYTE_E220_UART_BSP_VERSION_MAIN   (0x01) /*!< [31:24] main version */
#define __CX32L003_EBYTE_E220_UART_BSP_VERSION_SUB1   (0x00) /*!< [23:16] sub1 version */
#define __CX32L003_EBYTE_E220_UART_BSP_VERSION_SUB2   (0x00) /*!< [15:8]  sub2 version */
#define __CX32L003_EBYTE_E220_UART_BSP_VERSION_RC     (0x00) /*!< [7:0]  release candidate */ 
#define __CX32L003_EBYTE_E220_UART_BSP_VERSION       ((__CX32L003_EBYTE_E220_UART_BSP_VERSION_MAIN << 24) \
													|(__CX32L003_EBYTE_E220_UART_BSP_VERSION_SUB1 << 16) \
													|(__CX32L003_EBYTE_E220_UART_BSP_VERSION_SUB2 << 8 ) \
													|(__CX32L003_EBYTE_E220_UART_BSP_VERSION_RC))

/**
  * @}
  */ 


/** @defgroup CX32L003_EBYTE_E220_UART_Private_Variables CX32L003 EBYTE_E220_UART Private Variables
  * @{
  */ 
// No LED / BUTTON available
/*
GPIO_TypeDef* LED_PORT[LEDn] = {LED1_GPIO_PORT};
		
const uint16_t LED_PIN[LEDn] = {LED1_PIN};	
	
GPIO_TypeDef* BUTTON_PORT[BUTTONn]  = {USER_BUTTON_GPIO_PORT}; 
const uint16_t BUTTON_PIN[BUTTONn]  = {USER_BUTTON_PIN}; 
const uint8_t  BUTTON_IRQn[BUTTONn] = {USER_BUTTON_EXTI_IRQn };
*/

/**
 * @brief BUS variables
 */
static uint32_t SPI_Timeout = E220_UART_SPIx_TIMEOUT_MAX; /*<! Value of Timeout when SPI communication fails */
SPI_HandleTypeDef hspi;

#ifdef RESERVED_HAL_ADC_MODULE_ENABLED
static ADC_HandleTypeDef hstarterkit_Adc;
/* ADC channel configuration structure declaration */
static ADC_ChannelConfTypeDef sConfig;
#endif /* RESERVED_HAL_ADC_MODULE_ENABLED */

/**
  * @}
  */ 

/** @defgroup CX32L003_EBYTE_E220_UART_Exported_Functions CX32L003 EBYTE_E220_UART Exported Functions
  * @{
  */ 

/**
  * @brief  This method returns the CX32L003 EBYTE_E220_UART BSP Driver revision
  * @retval version : 0xXYZR (8bits for each decimal, R for RC)
  */
uint32_t BSP_GetVersion(void)
{
  return __CX32L003_EBYTE_E220_UART_BSP_VERSION;
}

/** @defgroup CX32L003_EBYTE_E220_UART_GPIO_Functions CX32L003 EBYTE_E220_UART GPIO Functions
  * @{
  */ 

/**
  * @brief  Configures Peripherals/GPIO for E220-900T22D.
  */
void BSP_GPIO_Init(void)
{
	GPIO_InitTypeDef gpio_init;
    /* Peripheral clock enable for all Peripherals/GPIO used in E220-900T22D */
    __HAL_RCC_GPIOA_CLK_ENABLE();
    __HAL_RCC_GPIOB_CLK_ENABLE();
    __HAL_RCC_GPIOC_CLK_ENABLE();
    __HAL_RCC_GPIOD_CLK_ENABLE();

    __HAL_RCC_UART0_CLK_ENABLE();
	__HAL_RCC_UART1_CLK_ENABLE();
	__HAL_RCC_SPI_CLK_ENABLE();

	/* RXD PA1 UART1 RX & TXD PA2 UART1 TX configured also in log.c */
    /* RXD UART1 RXD AFR7 */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = RXD_GPIO_PIN;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF7_UART1_RXD;
    HAL_GPIO_Init(RXD_GPIO_PORT, &gpio_init);
    /* TXD UART1 TXD AFR7 */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = TXD_GPIO_PIN;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF7_UART1_TXD;
    HAL_GPIO_Init(TXD_GPIO_PORT, &gpio_init);

    /* AUX GPIO OUT for debug AFR0 */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = AUX_GPIO_PIN;
    gpio_init.Mode = GPIO_MODE_OUTPUT;
    gpio_init.OpenDrain = GPIO_PUSHPULL;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.DrvStrength = GPIO_DRV_STRENGTH_HIGH;
    gpio_init.Pull = GPIO_NOPULL;
    HAL_GPIO_Init(AUX_GPIO_PORT, &gpio_init);

    /* LLCC68 DIO1 is used as IRQ (see call to llcc68_set_dio_irq_params()) */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = LLCC68_DIO1_GPIO_PIN;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.Pull = GPIO_NOPULL;
    /* Configure BUSY pin as input with External interrupt */
    gpio_init.Mode = EXTI_MODE;
    gpio_init.Exti.Enable = GPIO_EXTI_INT_ENABLE;
    gpio_init.Exti.EdgeLevelSel = GPIO_EXTI_INT_EDGE;
    gpio_init.Exti.RiseFallSel = GPIO_EXTI_INT_HIGHRISE;
    HAL_GPIO_Init(LLCC68_DIO1_GPIO_PORT, &gpio_init);

  	__HAL_GPIO_EXTI_CLEAR_FLAG(LLCC68_DIO1_GPIO_PORT, LLCC68_DIO1_GPIO_PIN);
    /* Enable and set DIO1 EXTI Interrupt to the lowest priority */
    HAL_NVIC_SetPriority(LLCC68_DIO1_GPIO_EXTI_IRQn, PRIORITY_LOW);
    HAL_NVIC_EnableIRQ(LLCC68_DIO1_GPIO_EXTI_IRQn);

    /* PC6 GPIO IN(safe) CX32L003 Test IO (Requires PCBite probe) */
    /*
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = GPIO_PIN_6;
    gpio_init.Mode = GPIO_MODE_INPUT;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.Pull = GPIO_NOPULL;
    HAL_GPIO_Init(GPIOC, &gpio_init);
    */

    /* LLCC68 BUSY IN */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = LLCC68_BUSY_GPIO_PIN;
    gpio_init.Mode = GPIO_MODE_INPUT;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.Pull = GPIO_NOPULL;
    HAL_GPIO_Init(LLCC68_BUSY_GPIO_PORT, &gpio_init);

    /* LLCC68 SX_NRESET OUT (3V3=1=No Reset / GND=0=Reset) */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = LLCC68_SX_NRESET_GPIO_PIN;
    gpio_init.Mode = GPIO_MODE_OUTPUT;
    gpio_init.OpenDrain = GPIO_PUSHPULL;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.DrvStrength = GPIO_DRV_STRENGTH_HIGH;
    gpio_init.Pull = GPIO_NOPULL;
    HAL_GPIO_Init(LLCC68_SX_NRESET_GPIO_PORT, &gpio_init);
    /* Reset the LLCC68 */
    HAL_GPIO_WritePin(LLCC68_SX_NRESET_GPIO_PORT, LLCC68_SX_NRESET_GPIO_PIN, GPIO_PIN_RESET); /* Reset Active low */

# if 0
    /* M0 PD3 UART0 RXD AFR5 */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = GPIO_PIN_3;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF5_UART0_RXD;
    HAL_GPIO_Init(GPIOD, &gpio_init);

    /* M1 PD4 UART0 TXD AFR5 */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = GPIO_PIN_4;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF5_UART0_TXD;
    HAL_GPIO_Init(GPIOD, &gpio_init);
#endif
    /* CX32L003F8 SPI configuration with LLCC68 */
    /* SPI MASTER NSS AFR3 for LLCC68 SPI SLAVE NSS */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = E220_UART_SPIx_NSS_PIN;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF3_SPI_NSS;
    gpio_init.OpenDrain = GPIO_PUSHPULL;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.DrvStrength = GPIO_DRV_STRENGTH_HIGH;
    gpio_init.Pull = GPIO_PULLUP;
    HAL_GPIO_Init(E220_UART_SPIx_NSS_GPIO_PORT, &gpio_init);

    /* PD5 SPI MASTER MISO AFR3 for LLCC68 SPI SLAVE MISO */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = E220_UART_SPIx_MISO_PIN;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF3_SPI_MISO;
    gpio_init.OpenDrain = GPIO_OPENDRAIN;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.DrvStrength = GPIO_DRV_STRENGTH_LOW;
    gpio_init.Pull = GPIO_PULLUP;
    HAL_GPIO_Init(E220_UART_SPIx_MISO_MOSI_GPIO_PORT, &gpio_init);

    /* PD6 SPI MASTER MOSI AFR3 for LLCC68 SPI SLAVE MOSI */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = E220_UART_SPIx_MOSI_PIN;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF3_SPI_MOSI;
    gpio_init.OpenDrain = GPIO_PUSHPULL;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.DrvStrength = GPIO_DRV_STRENGTH_HIGH;
    gpio_init.Pull = GPIO_PULLUP;
    HAL_GPIO_Init(E220_UART_SPIx_MISO_MOSI_GPIO_PORT, &gpio_init);
	HAL_GPIO_WritePin(E220_UART_SPIx_MISO_MOSI_GPIO_PORT, E220_UART_SPIx_MOSI_PIN, GPIO_PIN_SET);

    /* SPI MASTER SCK AFR3 for LLCC68 SPI SLAVE SCK */
    memset(&gpio_init, 0, sizeof(GPIO_InitTypeDef));
    gpio_init.Pin = E220_UART_SPIx_SCK_PIN;
    gpio_init.Mode = GPIO_MODE_AF;
    gpio_init.Alternate = GPIO_AF3_SPI_CLK;
    gpio_init.OpenDrain = GPIO_PUSHPULL;
    gpio_init.Debounce.Enable = GPIO_DEBOUNCE_DISABLE;
    gpio_init.SlewRate = GPIO_SLEW_RATE_HIGH;
    gpio_init.DrvStrength = GPIO_DRV_STRENGTH_HIGH;
    gpio_init.Pull = GPIO_PULLUP;
    HAL_GPIO_Init(E220_UART_SPIx_SCK_GPIO_PORT, &gpio_init);
}

/**
  * @brief SPI Init Configuration
  * @retval None
  */
void BSP_SPI_Init(void)
{
	/* SPI init for SPI LLCC68 */
    hspi.Instance               = SPI;
	hspi.Init.NSS               = SPI_NSS_MODE_HIGH;// SSN value controls the SPI_CS port level to be high
    hspi.Init.Mode              = SPI_MODE_MASTER;
    hspi.Init.CLKPolarity       = SPI_POLARITY_LOW;
    hspi.Init.CLKPhase          = SPI_PHASE_1EDGE;
    hspi.Init.BaudRatePrescaler = SPI_BAUDRATEPRESCALER_2;// Fsys/2 => 24/2= 12MHz
	HAL_SPI_Init(&hspi);

    /* Stop Reset the LLCC68 */
    HAL_GPIO_WritePin(LLCC68_SX_NRESET_GPIO_PORT, LLCC68_SX_NRESET_GPIO_PIN, GPIO_PIN_SET); /* Reset Active low */
}

/**
  * @brief SPI write data
  * @retval None
  */
void BSP_SPI_Write_Data(uint8_t *data, uint16_t data_size)
{
	uint8_t i;
	uint8_t r_data = 0;
	HAL_SPI_Set_NSS(&hspi, SPI_NSS_MODE_LOW);
	for(i = 0; i < data_size; i++)
	{
		HAL_SPI_Master_Send_Data(&hspi, data[i]);
	}
	HAL_SPI_Master_Receive_Data(&hspi, &r_data); // Flush last data
	HAL_SPI_Set_NSS(&hspi, SPI_NSS_MODE_HIGH);
}

/**
  * @brief SPI READ data
  * @retval None
  */
void BSP_SPI_Read_Data(uint8_t *data, uint8_t data_size)
{
	uint8_t i = 0, w_data = 0;

	HAL_SPI_Set_NSS(&hspi, SPI_NSS_MODE_LOW);
	for(i = 0; i < data_size; i++)
	{
		HAL_SPI_Master_Send_Data(&hspi, w_data);
		HAL_SPI_Master_Receive_Data(&hspi, &data[i]);
	}
	HAL_SPI_Set_NSS(&hspi, SPI_NSS_MODE_HIGH);
}

/**
  * @}
  */ 

/** @defgroup CX32L003_EBYTE_E220_UART_BUTTON_Functions CX32L003 EBYTE_E220_UART BUTTON Functions
  * @{
  */ 

/**
  * @}
  */ /* End of group CX32L003_EBYTE_E220_UART_Exported_Functions */

/** @addtogroup CX32L003_EBYTE_E220_UART_Private_Functions
  * @{
  */ 
  
/**
  * @}
  */
  
/**
  * @}
  */ /* End of group CX32L003_EBYTE_E220_UART */  

/**
  * @}
  */ /* End of group BSP */ 
     
    
